-- Digital Menu Board Database Schema
-- Run this SQL script to create your database

CREATE DATABASE IF NOT EXISTS digital_menu_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE digital_menu_db;

-- Users/Clients Table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    business_name VARCHAR(100) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    status ENUM('active', 'inactive') DEFAULT 'active',
    INDEX idx_username (username),
    INDEX idx_email (email)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Menu Boards Table (multiple boards per client)
CREATE TABLE IF NOT EXISTS menu_boards (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    board_name VARCHAR(100) NOT NULL,
    board_slug VARCHAR(100) UNIQUE NOT NULL,
    is_active BOOLEAN DEFAULT true,
    refresh_interval INT DEFAULT 30, -- seconds
    theme_config JSON, -- stores colors, fonts, layout settings
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_board_slug (board_slug)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Menu Categories Table
CREATE TABLE IF NOT EXISTS menu_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    board_id INT NOT NULL,
    category_name VARCHAR(100) NOT NULL,
    display_order INT DEFAULT 0,
    is_visible BOOLEAN DEFAULT true,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (board_id) REFERENCES menu_boards(id) ON DELETE CASCADE,
    INDEX idx_board_id (board_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Menu Items Table
CREATE TABLE IF NOT EXISTS menu_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    item_name VARCHAR(150) NOT NULL,
    description TEXT,
    price DECIMAL(10,2) NOT NULL,
    image_url VARCHAR(255),
    is_available BOOLEAN DEFAULT true,
    is_featured BOOLEAN DEFAULT false,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id) ON DELETE CASCADE,
    INDEX idx_category_id (category_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Scheduled Price Changes Table
CREATE TABLE IF NOT EXISTS price_schedules (
    id INT AUTO_INCREMENT PRIMARY KEY,
    item_id INT NOT NULL,
    scheduled_price DECIMAL(10,2) NOT NULL,
    start_time TIME NOT NULL,
    end_time TIME NOT NULL,
    days_of_week JSON NOT NULL, -- ["monday", "tuesday", etc.]
    is_active BOOLEAN DEFAULT true,
    schedule_name VARCHAR(100), -- e.g., "Happy Hour", "Lunch Special"
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (item_id) REFERENCES menu_items(id) ON DELETE CASCADE,
    INDEX idx_item_id (item_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Design Templates Table (for future Canva-like feature)
CREATE TABLE IF NOT EXISTS design_templates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    board_id INT NOT NULL,
    template_name VARCHAR(100) NOT NULL,
    template_data JSON NOT NULL, -- stores design elements, positions, animations
    is_active BOOLEAN DEFAULT false,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (board_id) REFERENCES menu_boards(id) ON DELETE CASCADE,
    INDEX idx_board_id (board_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Activity Log Table
CREATE TABLE IF NOT EXISTS activity_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    action_type VARCHAR(50) NOT NULL, -- 'price_update', 'item_add', 'item_delete', etc.
    action_details JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert demo data
INSERT INTO users (username, email, password_hash, business_name) VALUES 
('demo', 'demo@restaurant.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Demo Restaurant');
-- Password is: password123

INSERT INTO menu_boards (user_id, board_name, board_slug, theme_config) VALUES 
(1, 'Main Menu Board', 'demo-main-board', '{"primaryColor": "#FF6B35", "secondaryColor": "#F7931E", "fontFamily": "Poppins", "bgColor": "#1a1a1a"}');

INSERT INTO menu_categories (board_id, category_name, display_order) VALUES 
(1, 'Appetizers', 1),
(1, 'Main Courses', 2),
(1, 'Desserts', 3),
(1, 'Beverages', 4);

INSERT INTO menu_items (category_id, item_name, description, price, is_featured, display_order) VALUES 
(1, 'Buffalo Wings', 'Crispy wings tossed in our signature buffalo sauce', 12.99, true, 1),
(1, 'Mozzarella Sticks', 'Golden fried mozzarella with marinara sauce', 8.99, false, 2),
(2, 'Classic Burger', 'Angus beef patty with lettuce, tomato, and special sauce', 15.99, true, 1),
(2, 'Grilled Salmon', 'Fresh Atlantic salmon with seasonal vegetables', 24.99, true, 2),
(3, 'Chocolate Lava Cake', 'Warm chocolate cake with molten center', 7.99, false, 1),
(4, 'Fresh Lemonade', 'House-made lemonade', 3.99, false, 1),
(4, 'Craft Beer', 'Rotating selection of local craft beers', 6.99, false, 2);

-- Add a happy hour schedule (Mon-Fri, 4pm-6pm, 20% off wings)
INSERT INTO price_schedules (item_id, scheduled_price, start_time, end_time, days_of_week, schedule_name) VALUES 
(1, 9.99, '16:00:00', '18:00:00', '["monday", "tuesday", "wednesday", "thursday", "friday"]', 'Happy Hour Special');
